﻿namespace Microsoft.Samples.PlanMyNight.Infrastructure.Tests
{
    using System.Collections;
    using System.Text.RegularExpressions;
    using System.Web;
    using System.Web.Mvc;
    using Microsoft.Samples.PlanMyNight.Infrastructure.Mvc;
    using Microsoft.VisualStudio.TestTools.UnitTesting;
    using Moq;

    [TestClass]
    public class AjaxClientScriptExtensionsFixture
    {
        [TestMethod]
        public void ShouldRenderRegisteredReferences()
        {
            var ajaxHelper = CreateAjaxHelper();
            ajaxHelper.RegisterClientScriptInclude("/Scripts/myScript.js");

            string actual = ajaxHelper.RenderClientScripts();

            StringAssert.Contains(actual, "myScript.js");
            StringAssert.Contains(actual, "<script type=\"text/javascript\" src=\"/Scripts/myScript.js\"></script>");
        }

        [TestMethod]
        public void ShouldRenderRegisteredReferencesUsingCdn()
        {
            var ajaxHelper = CreateAjaxHelper();
            ajaxHelper.RegisterClientScriptInclude("/Scripts/myScript.js", "http://cdn.com/myScript.js");

            string actual = ajaxHelper.RenderClientScripts(true);

            StringAssert.Contains(actual, "myScript.js");
            StringAssert.Contains(actual, "<script type=\"text/javascript\" src=\"http://cdn.com/myScript.js\"></script>");
        }

        [TestMethod]
        public void ShouldRenderRegisteredCombinedReferences()
        {
            var ajaxHelper = CreateAjaxHelper();
            ajaxHelper.RegisterCombinedScriptInclude("/Scripts/myScript.js");
            ajaxHelper.RegisterCombinedScriptInclude("/Scripts/myScript2.js");

            string actual = ajaxHelper.RenderClientScripts();

            StringAssert.Contains(actual, "myScript.js");
            StringAssert.Contains(actual, "myScript2.js");
            StringAssert.Contains(actual, "<script type=\"text/javascript\" src=\"/JsCombine.ashx?js=%2fScripts%2fmyScript.js,%2fScripts%2fmyScript2.js\"></script>");
        }

        [TestMethod]
        public void ShouldRenderRegisteredCombinedReferencesUsingGroups()
        {
            var ajaxHelper = CreateAjaxHelper();
            ajaxHelper.RegisterCombinedScriptInclude("/Scripts/myScript.js", "group1");
            ajaxHelper.RegisterCombinedScriptInclude("/Scripts/myScript2.js", "group1");
            ajaxHelper.RegisterCombinedScriptInclude("/Scripts/myScript3.js", "group2");
            ajaxHelper.RegisterCombinedScriptInclude("/Scripts/myScript4.js", "group2");
            ajaxHelper.RegisterCombinedScriptInclude("/Scripts/myScript5.js");
            ajaxHelper.RegisterCombinedScriptInclude("/Scripts/myScript6.js");

            string actual = ajaxHelper.RenderClientScripts();

            StringAssert.Contains(actual, "myScript.js");
            StringAssert.Contains(actual, "myScript2.js");
            StringAssert.Contains(actual, "myScript3.js");
            StringAssert.Contains(actual, "myScript4.js");
            StringAssert.Contains(actual, "myScript5.js");
            StringAssert.Contains(actual, "myScript6.js");
            StringAssert.Contains(actual, "<script type=\"text/javascript\" src=\"/JsCombine.ashx?js=%2fScripts%2fmyScript.js,%2fScripts%2fmyScript2.js\"></script>");
            StringAssert.Contains(actual, "<script type=\"text/javascript\" src=\"/JsCombine.ashx?js=%2fScripts%2fmyScript3.js,%2fScripts%2fmyScript4.js\"></script>");
            StringAssert.Contains(actual, "<script type=\"text/javascript\" src=\"/JsCombine.ashx?js=%2fScripts%2fmyScript5.js,%2fScripts%2fmyScript6.js\"></script>");
        }

        [TestMethod]
        public void ShouldRenderRegisteredBlocks()
        {
            var ajaxHelper = CreateAjaxHelper();
            string script = "var test = function() { };";

            ajaxHelper.RegisterClientScriptBlock(script);

            string actual = ajaxHelper.RenderClientScripts();

            StringAssert.Contains(actual, script);
        }

        [TestMethod]
        public void ShouldRenderRegisteredBlocksInsideScriptTags()
        {
            var ajaxHelper = CreateAjaxHelper();
            string script = "var myCode;";

            ajaxHelper.RegisterClientScriptBlock(script);

            string actual = ajaxHelper.RenderClientScripts();

            StringAssert.Matches(actual, new Regex("<script.*var myCode;.*</script>", RegexOptions.Singleline));
        }

        [TestMethod]
        public void ShouldNotRenderTwiceTheSameRegisteredInclude()
        {
            var ajaxHelper = CreateAjaxHelper();
            string url = "duplicateUrl";

            ajaxHelper.RegisterClientScriptInclude(url);
            ajaxHelper.RegisterClientScriptInclude(url);

            string actual = ajaxHelper.RenderClientScripts();
            Assert.AreEqual(actual.IndexOf(url), actual.LastIndexOf(url));
        }

        private static AjaxHelper CreateAjaxHelper()
        {
            var httpContext = new Mock<HttpContextBase>();
            var dictionary = new Hashtable();
            httpContext.SetupGet(c => c.Items).Returns(dictionary);
            var httpRequest = new Mock<HttpRequestBase>();
            httpRequest.SetupGet(o => o.ApplicationPath).Returns("/");
            httpContext.SetupGet(c => c.Request).Returns(httpRequest.Object);
            var ajaxHelper = new AjaxHelper(new ViewContext { HttpContext = httpContext.Object }, new Mock<IViewDataContainer>().Object);
            return ajaxHelper;
        }
    }
}
